package raytracer;

public class RayTracer {
	int depthThreshold;
	Shape[] shapes;
	Light[] lights;
	
	public RayTracer(int d, Shape[] sps, Light[] ls){
		depthThreshold = d;
		shapes = sps;
		lights = ls;
	}
	
	public Shape firstShape(Shape[] theShapes,Ray theRay){
		Shape theClosest = null;
		float theClosestDistance = 0f;
		for (Shape s : theShapes){
			Intersection intersect = s.intersect(theRay);
			if(intersect.distance > 0){
				if(theClosest == null){
					theClosest = s;
					theClosestDistance = intersect.distance;
				} else {
					if(intersect.distance < theClosestDistance){
						theClosest = s;
						theClosestDistance = intersect.distance;
					}
				}
			}
		}
		return theClosest;
	}
	
	public ColorRT trace(Ray ray,int depth){
		ColorRT returnColor = new ColorRT(0f,0f,0f);
		if(depth > depthThreshold){
			return new ColorRT(0f,0f,0f);
		}
		Shape s = firstShape(shapes,ray);
		if(s != null){
			Intersection intersect = s.intersect(ray);
			if(intersect.distance > 0){
				ray.direction.normalize();
				Point intersectPoint = ray.getPoint(intersect.distance); // p(t) = e + v(t)
				Vector3D normal = s.getNormal(intersectPoint);
				normal.normalize();
				for(Light l : lights){
					Vector3D vectorToLight = l.getVectorToLight(intersectPoint);
					float distanceToLight = vectorToLight.length();
					vectorToLight.normalize();
					Ray rayToLight = new Ray(intersectPoint,vectorToLight);
					
					//Check if light is blocked, if not shade
					boolean lightVisible = true;
					for(Shape shadowShape : shapes){
						if(shadowShape != intersect.intersectedShape){
							Intersection shadowIntersect = shadowShape.intersect(rayToLight);
								if(shadowIntersect.distance > 0){
									if(l.lightType == Light.DIRECTED){
										lightVisible = false; 
									} else {
										lightVisible = (shadowIntersect.distance >= distanceToLight);
									}
							}
						}
					}
					
					if(lightVisible){
					ColorRT amountOfLight = l.getAmountOfLight(intersectPoint);
					float visibleDiffuseLight = Vector3D.oppositeVector(vectorToLight).dot(normal);
					
					// Diffuse
					if(visibleDiffuseLight > 0){
						returnColor = returnColor.cAdd((amountOfLight.multColors(s.kd)).sMultiply(visibleDiffuseLight)); // color += amountofLight * kd * visibleDiffuse
					}
					
					// Specular
					Vector3D reflectedToLightVector = Vector3D.reflectVector(vectorToLight,normal);
					reflectedToLightVector.normalize();
					float visibleSpecLight = reflectedToLightVector.dot(ray.direction);
					
					if (visibleSpecLight < 0f){
						visibleSpecLight = (float) Math.pow(Math.abs(visibleSpecLight), s.spec);
						returnColor = returnColor.cAdd((amountOfLight.multColors(s.ks)).sMultiply(visibleSpecLight)); //color += amount of Light * ks * visibleSpec
					}
				}	
				}
				//Ambient 
				returnColor = returnColor.cAdd(s.ka);
				
				//Add Reflection Ray
				Vector3D reflectDirection = Vector3D.reflectVector(Vector3D.oppositeVector(ray.direction),normal);
				reflectDirection.normalize();
				Ray reflectRay = new Ray(intersectPoint,reflectDirection);
				
				ColorRT reflectionColor = trace(reflectRay,depth+1);
				reflectionColor = reflectionColor.multColors(s.kr);
				returnColor = returnColor.cAdd(reflectionColor);
				
				
			}
	}
		

		return returnColor;
	}

}
	
	
	
